### A Pluto.jl notebook ###
# v0.20.4

using Markdown
using InteractiveUtils

# This Pluto notebook uses @bind for interactivity. When running this notebook outside of Pluto, the following 'mock version' of @bind gives bound variables a default value (instead of an error).
macro bind(def, element)
    #! format: off
    quote
        local iv = try Base.loaded_modules[Base.PkgId(Base.UUID("6e696c72-6542-2067-7265-42206c756150"), "AbstractPlutoDingetjes")].Bonds.initial_value catch; b -> missing; end
        local el = $(esc(element))
        global $(esc(def)) = Core.applicable(Base.get, el) ? Base.get(el) : iv(el)
        el
    end
    #! format: on
end

# ╔═╡ 419e9d2c-f5df-49a3-b0ce-9804aacda4b7
# ╠═╡ show_logs = false
import Pkg; Pkg.activate(".")

# ╔═╡ 6a59a8e6-39fc-44b3-9921-8976c677f4b1
begin
	using CairoMakie
	using CommonMark
	using CSV
	using DataFrames
	using Dates
	using Distributions
	using DSP
	using FFTW
	using Format
	using HTTP
	using LaTeXStrings
	using Latexify
	using PlutoUI
	using Statistics
end

# ╔═╡ d9329a8d-b07c-4207-93a6-1668da23e296
md"""
**What is this?**


*This jupyter notebook is part of a collection of notebooks on various topics discussed during the Time Domain Astrophysics course delivered by Stefano Covino at the [Università dell'Insubria](https://www.uninsubria.eu/) in Como (Italy). Please direct questions and suggestions to [stefano.covino@inaf.it](mailto:stefano.covino@inaf.it).*
"""

# ╔═╡ 3307644d-e875-4f15-a140-a41f7ca82a8f
md"""
**This is a `Julia` notebook**
"""

# ╔═╡ 7209ecae-fba8-44d7-940e-1ecc12e6700a
Pkg.instantiate()

# ╔═╡ 95ee75d5-3112-42b6-83aa-29e639ac6eb0
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/TimeDomainBanner.jpg"))
"""

# ╔═╡ b175b388-911f-4862-afdb-7449fec2bf9e
cm"""
# Time-Series
***

- A time-series is any sequene of observation such that the distribution of a given value depends on the previous values.
- Time is an exogeneous (outside the model) variable that is directional - measurements only depend on the past.
    - This is a statement of causality.

- However, the exogenous variable can be anything.

- Let's assume to have a set of data extracted from ``y(t) = A \sin(\omega t)`` with homoscedastic variance ``V = \sigma^2 + A^2/2``.
    - This is easy to prove if you compute the variance as ``\sum (y-\lt y \gt)^2 / N``. Since the average value is zero, this turns out to be ``V = \frac{A^2}{N} \sum \sin^2 (\omega t)`` giving the ``A^2/2`` term.

- We can compute the ``\chi^2`` for this toy model:
```math
\chi^2_{\rm dof} = \frac{1}{N} \sum (y/\sigma)^2 = \frac{1}{N \sigma^2} \sum (y)^2 = \frac{1}{\sigma^2} {VAR} = 1 + \frac{A^2}{2\sigma^2}
```

- With no variability (``A \sim 0``) the expectation value of the ``\chi^2_{\rm dof} \sim 1`` with standard deviation ``\sqrt{2/N}``, while it'll be larger in case of variability.
- Therefore, in order to have ``\chi^2_{\rm dof} > 1 + 3 \sqrt{2/N}`` we need ``A > \sigma \sqrt[4]{72/N}``, which shows that with ``N`` sufficiently large we can detect variability well below the uncetainty of the single points.
"""

# ╔═╡ 882fc480-b132-45a8-906a-db157059b92c
md"""
# Fourier Analysis
***

- The aim of Fourier analysis is to express any function as a sum of different sines and cosines, characterised by an angular frequency $ω$ or a corresponding time period $P = 2π/ω$.

- Such a decomposition is known as a Fourier series:

```math
f(t) = \frac{a(0)}{2}+\sum_{n=1}^\infty [ a(n)\sin n\omega_0 t + b(n)\cos n\omega_0 t]
```

- This expression can represent practically any periodic function with period $P_0$, with suitable adjustment of the coefficients $a(n)$ and $b(n)$, which are known as the Fourier coefficients.

- The conditions under which the Fourier decomposition is valid are that $f(t)$ has only a finite number of finite discontinuities and only a finite number of extreme values within a period.

    - These are known as Dirichlet conditions and the functions obeying them are called piece-wise regular.

- The Fourier coefficients $a(n)$ and $b(n)$ can be determined by performing the following integrations:

```math
a(n) = \frac{2}{P_0} \int_0^{P_0} f(t)\sin n\omega_0 t \, dt \;\; ; \;\;\; b(n) = \frac{2}{P_0} \int_0^{P_0} f(t)\cos n\omega_0 t \, dt, \;\; n=0,1,2...
```

- Alternatively, the Fourier decomposition may also be expressed in an equivalent exponential form:

```math
f(t) = \frac{1}{P_0}\sum_{n=-\infty}^{\infty} c(n) e^{i n\omega_0 t}; \;\;\mbox{\rm where}\;\; c(n)=\int_{0}^{P_0} f(t) e^{-in\omega_0 t} dt
```

- In the above expansions, the $n = 0$ term is often called the constant or the D.C. (Direct Current) component, the $n = 1$ term the fundamental and the terms with $n > 1$ the harmonics.
"""

# ╔═╡ 231efd02-fbde-4c81-9577-7fea1ac07e88
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/squarewave.jpg"))
"""

# ╔═╡ bc82a078-fea7-4a00-b38c-a849fb760594
md"""
- Example of a Fourier series decomposition. The original time domain function $f(t)$, shown in the thick black line, is a square wave of period $T$, which equals unity from $t=0$ to $0.5T$ and zero from $0.5T$ to $T$. Two periods of the function are plotted. Thin lines show the Fourier sum with different number of terms: (i) black: DC$+$Fundamental, (ii) blue: sum up to 3rd harmonic, (iii) red: sum up to 11th harmonic. The gradual improvement in the approximation of the function with increasing number of terms in Fourier series is evident. At the point of discontinuity, all the reconstructions pass through the average of the left and right limits of the original function and the transition gets progressively sharper with larger number of terms.
"""

# ╔═╡ 3bdf3bea-7e3c-4f09-98f7-bdbc16ea2880
md"""
### Continuous Fourier Transform
***

- We define the Fourier transform (FT) of any function $f(t)$ as:

```math
F(\omega) = \int\limits_{-\infty}^{+\infty} f(t) e^{-i\omega t} dt
```

- This is a linear transformation and no information is lost. The representations of a function in time and frequency domains are equivalent.

- The original $f(t)$ can be recovered by applying the inverse Fourier transform:

```math
f(t) = {1\over 2\pi} \int\limits_{-\infty}^{+\infty} F(\omega) e^{i\omega t} d\omega
```

- The FT has a number of interesting properties. It is linear, not necessarily a real function, and its amplitude is invariant to time shift (but not its phase).

| $f(t)$ | F(ω) |
|:-------: |:--------:|
| Real    | H(-ω) = H$^*$(ω)   |
| Even    | Even   |
| Odd    | Odd   |
| Real and Even | Real and Even     |
| Real and Odd  | Imaginary and Odd |

- Unless the original function is even, an unlikely situation in the case of a time series, its Fourier transform is complex!
"""

# ╔═╡ 0dbb5f0e-292d-4d8a-bb83-7954e4a46f29
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/FTexamples.png"))
"""

# ╔═╡ 875732fc-5644-4c06-8f8c-39dc30f1d3b4
md"""
- Scaling: $h(at) \longleftrightarrow H(f/a) / |a|$, "broad" $\longleftrightarrow$ "narrow"
- Shifting: $h(t-t_0) \longleftrightarrow H(f) * e^{2\pi i f t_0}$, "shift" $\longleftrightarrow$ "phase roll/gradient"
- Convolution: $h(t) \ast g(t) \longleftrightarrow H(f) G(f)$, "convolution" $\longleftrightarrow$ "multiplication"
"""

# ╔═╡ 56081488-4ade-436a-991e-d7138368edf1
md""" 
> How to compute the Fourier Transform in a few simple cases can be found [here](./open?path=Lectures/Lecture - Spectral Analysis/Lecture-FT.jl).
"""

# ╔═╡ 6d88e888-8f08-47ab-8906-c7b4b86b2588
md"""
## Power density spectrum (PDS)
***

- The Power Density Spectrum (PSD) is defined as the Fourier Transform multiplied by its complex conjugate and therefore the square modulus of the Fourier Transform:

```math
P(\omega) = F(\omega)\cdot F^*(\omega) =  | F(\omega) |^2
```

- If the original function is real (usually the case for time series) the PSD is an even function and the values at negative frequencies are redundant.

- The FT is a linear function, the PDS is not. This means that while the FT of the sum of two signals is the sum of the FT of the signals, in the case of the PDS this is not true and there are cross-terms to be considered.

- E.g., if two signals are $f(t)$ and $g(t)$, the PSD of the sum of the two is:

```math
P[f(t)+g(t)] = |F[f(t)+g(t)]|^2 =  F[f(t)+g(t)]\cdot F^*[f(t)+g(t)] = 
```

```math
= P[f(t)] + P[g(t)] + 2 Re\{F[f(t)]\cdot F[g(t)] \}
```

- If the two signals are uncorrelated, the cross-term is zero and linearity applies.
"""

# ╔═╡ 99f8ea18-5807-482e-8516-e2bdaf1546e6
md"""
## Autocorrelation function (ACF)
***

- The autocorrelation (ACF) of a function $f(t)$ is defined as:

```math
A(t) = \int\limits_{-\infty}^{+\infty} f(\tau) f(t+\tau) d\tau \Longleftrightarrow F(f) F^*(f) \equiv |F(f)|^2
```

- The autocorrelation of a function is the Fourier transform of its PSD.

- It is also simple to derive, by the *Parseval’s theorem*, simply setting $t=0$, that:

```math
\int\limits_{-\infty}^{+\infty} |f(t)|^2 dt = {1\over 2\pi} \int\limits_{-\infty}^{+\infty} |f(\omega)|^2 d\omega
```

> For the interested readers, a proof that PDS and ACF are Fourier duals can be found [here](./open?path=Lectures/Lecture - Spectral Analysis/Lecture-PDS-ACF.jl).

"""

# ╔═╡ e087bf1e-dd1a-41f3-8154-bfb39325d905
cm"""
## Discrete Fourier transform
***

- In the real world, we only have discrete measurements, the time series, commonly called, in astronomy, “light curves”. They consist of ``N`` measurements ``x_k`` taken (now) at equally-spaced times ``t_k`` from ``0`` to ``T``.

- In this case we can define the discrete Fourier transform (and its inverse) as:

```math
a_j = \sum\limits_{k=0}^{N-1} x_k e^{-2\pi ijk/N}   \quad   (j=-N/2,...,N/2-1)
```

```math
x_k = {1\over N} \sum\limits_{k=-N/2}^{N/2-1} a_j e^{2\pi ijk/N} \quad\quad (k=0,...,N-1)
```

- Since the data are equally spaced, the times are ``kT/N`` and the frequencies are ``j/T``.

- The time step is ``δt = T/N`` and the frequency step is ``δν = 1/T``.

- As the discrete time series has a time step ``δt`` and a duration ``T``, there are limitations to the frequencies that can be examined:
    - The lowest frequency is ``1/T``, corresponding to a sinusoid with a period equal to the signal duration.
    - The highest frequency that can be sampled, is called *Nyquist frequency*: ``\nu_\rm{Nyq} = \frac{1}{2\delta T} = \frac{1}{2}\frac{N}{T} ``.

- At the zero frequency, the FT value is just the sum of the signal values:

```math
a_0 = \sum\limits_{k=0}^{N-1} x_k e^{-2\pi i0k/N} = \sum\limits_{k=0}^{N-1} x_k
```

- *Parseval’s theorem* applies also to the discrete case and one can see that the variance of a signal is ``1/N`` times the sum of the ``a_j`` over all indices besides zero (also known as *Plancherel Theorem*):

```math
Var(x_k) = \sum_k (x_k - \bar{x})^2 = \sum_k x_k^2 + \sum_k \bar{x}^2 - \sum_k 2\bar{x}x_k = \sum_k x_k^2 + N \bar{x}^2 -  2N\bar{x}^2 = 
```

```math
= \sum_k x_k^2 - N\bar{x}^2 = \sum_k x_k^2 - \frac{1}{N}(\sum_k x_k)^2 = \frac{1}{N} \sum_j |a_j|^2 - \frac{1}{N}a_0^2 =
```

```math
\Longrightarrow Var(x_k) = \frac{1}{N}\sum_{j=-\frac{N}{2}}^{j=\frac{N}{2}-1} |a_j|^2,  j\ne0
```

"""

# ╔═╡ de228ddb-cacc-456f-b6c7-464e81d1e1e8
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/power_of_PDS.png"))
"""

# ╔═╡ e82d615c-5201-4efb-8b77-37a5aa636f39
md"""
- Top panel: the black line shows 40 seconds of a simulated 1000s time series consisting of a weak sinusoidal modulation (red line) "drowned" into a strong Gaussian noise. Bottom panel: corresponding PDS, zoomed on the relevant frequency range, where the modulation is clearly visible. A weak signal spread in time is collected into a single frequency bin at high significance.
"""

# ╔═╡ 58d43f9c-e7e9-4a7d-8471-1beca3568a1e
md"""
### Exercize about an analysis of weather data in France
***

- In the following exercize, we are going to analyse weather data spanning about 20 years in France obtained from the US National Climatic Data Center.

- Data are imported [http://www.ncdc.noaa.gov/cdo-web/datasets#GHCND](http://www.ncdc.noaa.gov/cdo-web/datasets#GHCND). The number "-9999" is used for N/A values. And we need to parse dates contained in the DATE column
"""

# ╔═╡ 3d75fecb-157b-408a-aa32-2d2096a766ba
begin
	url = "https://github.com/ipython-books/cookbook-2nd-data/blob/master/weather.csv?raw=true"
	#fname = "France_temperatures.csv"
	
	df = DataFrame(CSV.File(HTTP.get(url).body,missingstring="-9999"))
	#df = DataFrame(CSV.File(fname,missingstring="-9999"))
	
	df[!,:DateTime] = Date.(string.(df[!,:DATE]),DateFormat("yyyymmdd"))
end;

# ╔═╡ 2e2e09aa-6023-46a9-b816-bb9de2b321c1
md"""
- Let's now select only dates later than Jan 1994, and compute daily averages dropping the missing data and plot our dataset.

    - The temperature unit is in tenths of a degree, and we get the average value between the minimal and maximal temperature.
"""

# ╔═╡ 0148c42e-24fa-486e-8c61-eae25e772bd8
begin
	filter!(:DateTime => >=(Date("19940101",DateFormat("yyyymmdd"))),df)
	
	dropmissing!(df)
	
	gdf = combine(groupby(df, :DateTime), [:PRCP,:TMAX,:TMIN] .=> mean, renamecols=false)
	
	temp = (gdf[!,:TMAX] + gdf[!,:TMIN]) / 20.
	N = length(temp)
	
	
	fg1 = Figure()
	
	ax1fg1 = Axis(fg1[1, 1],
	    xlabel = "Date (yyyy/mm/dd)",
	    ylabel = L"Average daily temperature ($^\circ$C)"
	    )
	
	scatter!(gdf[!,:DateTime],temp,color=:blue)
	
	fg1
end

# ╔═╡ 7709a017-0c80-4399-9751-bf07b80f0b5e
md"""
- We now compute the Fourier transform and the spectral density of the signal using the **fft()** function.
- Once the FFT has been obtained, we take the square of its absolute value in order to get the **power spectral density (PSD)**.
- Then, we get the frequencies corresponding to the values of the PSD by the **fftfreq()** utility function.  
- Since the original unit is in days we change it to annual unit by the factor **1/365**.
"""

# ╔═╡ b49f1cda-582a-4d43-9e0a-a0e6adf85e78
begin
	# The FFTW library has been used for this exercize
	
	temp_fft = fft(temp)
	
	temp_psd = abs.(temp_fft).^2
	
	# 1 is the sampling frequency
	temp_freq = fftfreq(length(temp_psd), 1) * 365
end;

# ╔═╡ c601e37c-c08d-4ce4-8843-6fd52bd1e812
md"""
- We restrict to positive frequences only and let's check which is the maximum frequency for our dataset. Knowing the sampling rate we predict that it should correspond to a period of 2 days so that the Nyquist frequency turns out to be exactly 0.5 day$^{-1}$.
"""

# ╔═╡ aff12d14-4803-480e-aab0-6d33917304b3
begin
	posfr = temp_freq .> 0
	
	#println("Nyquist frequency: ", round(maximum(temp_freq) / 365, digits=1), L" day$^{-1}$")
	
end;

# ╔═╡ 6f29671a-048a-49e1-b7f9-e5d9cd733a0c
Markdown.parse("""
##### Nyquist frequency:  $(latexify(maximum(temp_freq) / 365,fmt="%.1f"))

""")

# ╔═╡ b8a2074a-490e-4e92-ba59-3735184d7c91
md"""
- Let's now plot the power spectral density of our signal, as a function of the frequency (in unit of **1/year**). We choose a logarithmic scale for the y axis (decibels).
"""

# ╔═╡ 40fe7f7d-8d30-40f1-b6fa-932374c93380
begin
	fg2 = Figure()
	
	ax1fg2 = Axis(fg2[1, 1],
	    xlabel = "Frequency (1/year)",
	    ylabel = "PSD (dB)",
	    )
	
	lines!(temp_freq[posfr],10 * log10.(temp_psd[posfr]),color=:blue)
	
	xlims!(0,maximum(temp_freq[posfr]))
	
	fg2
end

# ╔═╡ 1eb98eee-8373-4fce-b40f-3118d096f082
md"""
- Or with a better zoom in a region of our interest.
"""

# ╔═╡ 0f5adef3-5de0-4fa3-b4ca-8ee59968047e
md"""
Plot horizontal axis limit: $( @bind xmx PlutoUI.Slider(0.1:0.5:200, default=7) ) 
"""

# ╔═╡ 1b13be55-5153-4d53-ba09-92832f67a448
begin
	fg3 = Figure()
	
	ax1fg3 = Axis(fg3[1, 1],
	    xlabel = "Frequency (1/year)",
	    ylabel = "PSD (dB)",
	    )
	
	lines!(temp_freq[posfr],10 * log10.(temp_psd[posfr]),color=:blue)
	
	xlims!(0,xmx)
	ylims!(30,85)
	
	fg3
end

# ╔═╡ bf2f9df2-7df7-43d2-926d-7a561be79c0e
md"""
- Not surprisingly, the fundamental frequency of the signal is the yearly variation of the temperature at **f=1**.

- We can now "clean" our data cutting out frequencies higher than the fundamental frequency and by an **inverse FFT** we recover a signal that mainly contains the fundamental frequency.
"""

# ╔═╡ bea9ec97-57a7-45a4-9a29-ccca274fd5dc
begin
	temp_fft_bis = copy(temp_fft)
	temp_fft_bis[abs.(temp_freq) .> 1.1] .= 0
	
	temp_slow = real.(ifft(temp_fft_bis))
	
	
	l = gdf[!,:DateTime] .< Date(2000,1,1)
	
	
	fg4 = Figure()
	
	ax1fg4 = Axis(fg4[1, 1])
	
	scatter!(gdf[!,:DateTime][l],temp[l],color=:blue, label="Original data")
	lines!(gdf[!,:DateTime][l],temp_slow[l],color=:red, label="Filtered data")
	
	ylims!(-10,40)
	
	axislegend()
	
	fg4
end

# ╔═╡ c1363e2a-d8b4-4be9-8394-435231b62177
# ╠═╡ show_logs = false
cm"""
### Windowing and Sampling
***

- The CFT and the DFT can be connected easily taking into account that the FT of the product fo two functions is the convolution of the FT of the functions.

```math
F[x \cdot y] = F[x] \otimes F[y] = \int\limits_{-\infty}^{+\infty}F[x(\nu')]F[y(\nu-\nu')] d\nu'
```

- A discrete time series `x(t_k) ≡ x_k` can be seen as the product of a continuous function `f(t)` over `(−∞,∞)` and two additional functions: `w(t)` to limit it to the `(0,T)` interval and `s(t)` to sample it at times tk:

```math
x_k = h(t) \cdot w(t) \cdot s(t)
```

- ``w(t)`` is a boxcar window function, which is 1 in the ``(0,T) `` interval and zero outside. ``s(t)`` is a series of delta functions at ``t_k``, spaced by ``T/N``:

$(LocalResource("Pics/windowing.png"))
"""

# ╔═╡ f5fea622-f431-4da6-af3f-548fd10d906d
# ╠═╡ show_logs = false
md"""
### Windowing effects
***

- Let us consider a purely sinusoidal function $f(t) = sin(ωt)$, whose FT is a delta function at $ω$.

- The multiplication by the window function corresponds to the convolution of the delta function with the FT of the window.

- It is simple to calculate the FT of the window: we consider a window function that is unity in the $−T/2,T/2$ interval, as it is a real and even function, whose FT is also real and even:

```math
F(w(t)) = 2 {\sin(\pi\nu T)\over\pi\nu}
```

- This is the well known “sinc” function.

- An important general rule is that the FT peak is broader for shorter T. Something easily deducible from the general propertieds of FT.

    - The resolution of the signal FT is therefore higher the longer the observation is.
    
- In addition to the broadening, there is the formation of side lobes. They are much lower than the central peak, but cannot always be ignored.

$(LocalResource("Pics/window_ft.png"))


"""

# ╔═╡ 43caa1a2-3083-414d-9681-4153ade8a92b
md"""
### Sampling effects: aliasing
***

- The FT of a series of regularly spaced delta functions with spacing $T/N$ is itself a series of delta functions with spacing $N/T$:

```math
s(t)    = \sum\limits_{k=-\infty}^{+\infty} \delta(t - {kT\over N}) \Longleftrightarrow
    F(s(t)) = \sum\limits_{m=-\infty}^{+\infty} \delta(\nu - {mN\over T})
```
    
- Therefore, the effect of sampling on the FT of a sinusoidal signal with frequency $ν_0$ (a delta function at $ν_0$) is that of adding an infinite sequence of delta functions spaced by $N/T$, called *aliases*.

- Depending on the frequency of the original signal and the Nyquist frequency we can have different situations.
    - In fact, features at $ν = ν_{N/2} + ν_x$ also appear at $ν = ν_{N/2} − ν_x$.

- This happens because the transition from the CFT to the DFT involves two operations:
    - windowing, a convolution with the function $W(f)$, which is essentially a peak with a width $δf = 1/T$ plus sidelines,
    - and aliasing, a reflection of features above the Nyquist frequency back into the range ($0,ν_{N/2}$).
"""

# ╔═╡ dd78a7e2-f63e-4e84-af31-286d979074f5
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/aliases.png"))
"""

# ╔═╡ 95095834-abd5-43e6-aad0-cdfa402b0366
md"""
- Top panel: Aliasing on a sinusoidal signal at 15 Hz sampled at 40 Hz. In black the true signal FT amplitude, in red the aliased one. The blue region marks the interval below $\nu_{Nyq}$. The signal is detected and the aliases are not.
- Bottom panel: same as the top panel, but with a signal at 35 Hz. Here the signal is not detected, but the 5 Hz alias is.
"""

# ╔═╡ 1cfc40d3-24a3-4a5c-8538-61c13542b403
# ╠═╡ show_logs = false
cm"""
- We have all experienced aliasing effects when looking at fast rotating objects like an air fan under fluorescent light.
    - The light provides a sampling at 50 Hz (or 60 Hz, depending on where you live), while the fan has a periodicity.
    - Depending on its angular speed, you can see it rotating apparently much slower, or even to stop and rotate in the opposite direction.

$(LocalResource("Pics/sampling.png"))

- Time domain example of aliasing. The blue signal has ``\nu_0 = 0.1`` Hz. If it is sampled with ``\nu_{Nyq} = 0.038`` Hz (red points) the red dashed alias is the best fit to the data, with ``\nu_a = 0.023`` Hz.
"""

# ╔═╡ e8786a24-4391-40ad-a9ba-a0cb3f7bb2b0
md"""
- In the real world we do not really sample signals, but integrate them over finite time bins, i.e. we convolve it with a binning function:

```math
b(t) = \begin{cases}
{N\over T} & t\in [-{T\over 2N},{T\over 2N}]\\
0                    & {\rm outside}
\end{cases}
```

- Therefore, the signal FT will be multiplied by that of the binning function, which is again a sinc function:

```math
B(\nu) = {\sin \pi\nu /2\nu_{Nyq}\over \pi\nu /2\nu_{Nyq}}
```

- $B(ν)$ is a broad function that reaches $0$ at $2\nu_{Nyq}$ and has the value of $2/π$ at $\nu_{Nyq}$.
"""

# ╔═╡ 39e1ee25-a2f4-4e16-9fec-4a4e9c2c653e
md"""
### Effects of binning on the harmonic analysis
***

- Binning is a common operation carried out for different reasons, e.g., for casting an irregularly sampled time series to a regular grid, for improving the S/N of each point, to reduce the computational burden, etc.

> Nevertheless, binning is never a price-free operation.

- Let's assume we have a dataset defined as $D = \{y_1, y_2,..., y_n \}$, with observations carried out at a regular time step. Its discrete Fourier transform is:

```math
Y(\omega) \equiv \sum_{t=1}^n y_t e^{i \omega t}
```

- This is defined for continuous values of $\omega$ but we know that no loss of data happens if we compute the Fourier transform at the Fourier grid $\omega_k \equiv 2 \pi k / n, \quad 0 \le k < n$.


#### Moving average
***

- Let's suppose our data are replaced by a moving average of past values (i.e. a rebinning):

```math
z_t \equiv \sum_{s=0}^{m-1} y_{t-s} \omega_s
```

- where $\omega_s$ is the weighting coefficient for lag $s$.

- Since this can be seen as a convolution of the input data we know that the Fourier transform of $z$ is the product of the Fourier transform of the original data and of the average function:

```math
Z(\omega) = W(\omega) Y(\omega)
```

- where $W(\omega) = \sum_{s=0}^{m-1} \omega_s e^{i \omega s}$.

- In particular, for uniform weighting, $w_s = 1/m, 0 \le s < m$, and we have the well known ["sinc"](https://en.wikipedia.org/wiki/Sinc_function) function:

```math
W(\omega) = \frac{1}{m} \sum_{s=0}^{m-1} e^{-i \omega s} = e^{-i\frac{\omega}{2}(m-1)} \left[\frac{\sin(m\omega/2)}{m \sin(\omega/s)}\right]
```

- It is clear, therefore, that together with changing the phase of the Fourier transform, any binning typically decreases the amplitude of the Fourier transform of the original data: i.e. it might be useful for plotting purposes and for saving computer power, yet binning should typically be avoided.
"""

# ╔═╡ 136b29c5-ac92-4e32-878e-3ad99216f78d
md"""
### Window carpentry
***

- Having a longer observation reduces the width of the main window peak, but does not change the possible spillover effects.

- In some cases it can be advantageous to multiply the data by another window, not boxcar-shaped.

- This results in a loss of signal, as some data are multiplied by a factor less than unity, but there are advantages, depending on the chosen window.

    - Many window functions with different characteristics have been designed and one can tailor them depending on what is needed.

- The main features that identify a window in its PDS are: the width of the main peak $∆ω$, the relative amplitude of the first side lobe $L$ (expressed in decibels) and the slope of the decay of side lobes n:
"""

# ╔═╡ d490adbd-376a-403b-adc3-3ab4d4e65bc5
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/window_features.png"))
"""

# ╔═╡ 848e6f85-e3b9-4928-aed0-b316f97d18db
md"""
- The boxcar window is the one with the lowest $∆ω$, but with alternative windows it is possible to obtain a significant reduction of the amplitude of the side lobes.

| Window   | $\Delta\omega$ | L     | n  | Function                       |
|:--------:|:-------------: |:-----:|:--:|:------------------------------:|
| Boxcar   |     0.89       | -13db | 2  |         1                      |
| Hamming  |     1.36       | -43db | 2  | 0.54+0.46 cos(2πt)             |
| Hann     |     1.44       | -32db | 5  | 0.5(1-cos(2πt))                |
| Blackman |     1.68       | -58db | 5  | 0.42+0.5cos(2πt)+0.08cos(4πt)  |
| Gaussian |     1.55       | -56db | 2  | $\exp(-4.2 x^2$)               |
"""

# ╔═╡ 8d5c7719-647a-4e7e-905f-43051bb26716
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/windows.png"))
"""

# ╔═╡ a9964f9d-1f3a-493b-9ab8-a8933ff51c72
md"""
- Left panel: the shape of five windows: boxcar, Hamming, Hann, Blackman and Gaussian. Right panel: the corresponding PDS, corresponding to (half) the shape of the PDS of a sinusoidal signal. The sidelobes of all but the boxcar window are too low to be seen.

"""

# ╔═╡ a0392d53-9e0d-4d70-a9a5-b99de7884f9d
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/wtot.png"))
"""

# ╔═╡ eae97444-2431-448c-abe6-c14aa64fa409
md"""
- Each of the four panels contains a signal (top) and its PDS (bottom). In all four, the signal consist of Gaussian noise plus a sinusoid at P=200 s, with 1-s binning.
    - Top left: continuous exposure of $10^5$ s.
    - Top right: continuous exposure of $10^4$ s.
    - Bottom left: exposure of $10^4$ s split into three intervals. The PDS is computed including the gaps as consisting of points at zero level.
    - Bottom right: exposure of $10^4$ s split into three intervals as in the previous case. The gaps are filled with Gaussian noise.
"""

# ╔═╡ 8d3eb863-5cb0-4221-8fc8-c48361615e95
md"""
### Fast Fourier transform
***

- Evaluation of the Discrete Fourier Transform (DFT) of $N$ samples involves $\sim N^2$ multiplication and addition operations -- for every Fourier component $a_j$, each of the $N$ samples $x_k$ needs to be multiplied by a phase factor $e^{-2\pi ijk/N}$ and then they have to be summed.  

- The Fast Fourier Transform algorithm has been devised to accomplish this computation in much fewer steps, typically with $\sim N\log_2 N$ multiplications and additions.  This provides enormous computational savings for large transforms, and has made Fourier analysis accessible to cases where it would have been otherwise prohibitive.
"""

# ╔═╡ 0b0b8578-4c67-463a-bd53-b3e6bd1f9712
md"""
### PSD normalization
***

- The FT is a linear transformation and the PSD is its squared modulus, the PSD then scales with the square of the intensity level of the signal.

- It is possible to normalize the PSD in different ways. One of the most common is the so-called *Leahy normalization*:

```math
P_j^{Leahy} = {2\over N_\gamma} |a_j|^2
```

- where $N_\gamma$ is the total number of photons in the signal. If instead of counts we have fluxes, and noise is $\mathcal{N}(0,σ)$, $N_\gamma$ is substituted by $N_\rm{data}σ^2$.

- The periodogram with this normalization is also known as *Classical* or *Schuster* periodogram.

- With the Leahy normalization the total variance can also be written as:

```math
Var(x_k) = \frac{1}{N}\sum_{j=-\frac{N}{2}}^{j=\frac{N}{2}-1} |a_j|^2 \ (j\ne0) =  \frac{N_\gamma}{N} \left( \sum_{j=1}^{j=\frac{N}{2}-1} P_j + \frac{1}{2} P_{N/2}\right)
```

- This normalization leads to a known statistical distribution of signal power:
    - if the signal is dominated by fluctuations due to Poisson statistics and if $N_\gamma$  is large, powers follow a $\chi^2$ distribution with 2 degrees of freedom, $<P>=2$ and $Var(P)=4$.
- The reason is that the periodogram is the sum of the squares of the real and imaginary parts of the FT and, for a stochastic process, the latter are normally distributed, so the sum of their squares is distributed as a $\chi^2$ with 2 degrees of freedom.

- For other noise distributions (Poisson, etc.), for large N, due to the central limit theorem the real and imaginary parts become still normal.


"""

# ╔═╡ 8e458457-f940-460a-8638-debe1f94defc
md"""
- Periodograms are intrinsically very noisy. If the signal is divided into $S$ segments and the resulting PDS are averaged and rebinned by a factor $M$, the powers will be distributed as a $\chi^2$ with $2SM$ degrees of freedom scaled by $1/SM$: therefore, the average power remains $<P>=2$, but the variance is now $Var(P)=4/SM$.
    - The technique of dividing the time series into equal-duration intervals and averaging the corresponding PSD is called *Bartlett’s method*.

- The reduction in time duration $T$ increases the minimum frequency in the PSD $ν_\rm{min} = 1/T$.

- This method, in principle, also allows one to skip over (short) data gaps, which have dramatic effects on the PSD.


"""

# ╔═╡ b2e2bace-7e50-41f0-87fb-780737ff6616
md"""
### Exercise about PSD manipulation
***

- Let's generate two arrays of relative timestamps, one 8 seconds long and one 1600 seconds long, with dt = 0.03125 s, and make two signals in units of counts. The signal is a sine wave with amplitude = 300 cts/s, frequency = 2 Hz, phase offset = 0 radians, and mean = 1000 cts/s. We then add Poisson noise to the light curve and plot the shortest of the two.
"""

# ╔═╡ 1112b278-802d-4611-8954-2795db178e88
begin
	pf = @bind pl_fr NumberField(1:10, default=2)
	af = @bind am_fr NumberField(100:100:1000, default=300)
end;

# ╔═╡ 19a6c251-adc1-422d-a702-20ea9b971449
cm"Frequency for the test signal:"

# ╔═╡ a8b99222-18a7-4b47-8f98-8961a708ec57
pf

# ╔═╡ 77b0596d-8c4a-4d3d-9bd8-54d3fde618c8
cm"Amplitude for the test signal:"

# ╔═╡ d542c639-1c93-4edd-9132-facd18fc89fb
af

# ╔═╡ b96525bf-90e5-47b3-b305-1db0ad675c2c
begin
	dt = 0.03125  # seconds
	exposure = 8.  # seconds
	long_exposure = 1600. # seconds
	times = range(start=0, stop=exposure-dt, step=dt)  # seconds
	long_times = range(start=0, stop=long_exposure-dt, step=dt)  # seconds
	
	signal = am_fr .* sin.(pl_fr .* pi .* times ./ 0.5) .+ 1000  # counts/s
	long_signal = am_fr .* sin.(pl_fr .* pi .* long_times ./ 0.5) .+ 1000  # counts/s
	
	noisy = [rand(Poisson(theta)) for theta in signal .* dt]  # counts
	long_noisy = [rand(Poisson(theta)) for theta in long_signal .* dt]  # counts
	
	fg5 = Figure()
	
	ax1fg5 = Axis(fg5[1, 1],
	    xlabel = "Time (s)",
	    ylabel = "Counts (cts)",
	    )
	
	lines!(times,noisy,color=:blue,label="Noisy signal")
	lines!(times,signal .* dt,color=:green,label="Signal")
	
	axislegend()
	
	fg5
end

# ╔═╡ ae8a4ebf-da8d-4778-ad9a-87e3aab72399
md"""
- Now let's compute the periodogram
"""

# ╔═╡ b7213b38-21ac-4fb5-ab53-694d3ac2fda9
pf

# ╔═╡ 78dc57ce-1cbd-4f9c-9741-54bf430b9186
af

# ╔═╡ 25fcf199-7da4-4632-a2fa-d0131f58d533
begin
	# We use the DSP package
	
	psd = periodogram(noisy; fs=1/dt)
	
	fg6 = Figure()
	
	ax1fg6 = Axis(fg6[1, 1],
	    yscale = log10,
	    xlabel = "Frequency (Hz)",
	    ylabel = "Power",
	    )
	
	lines!(psd.freq,psd.power,color=:blue,label="PSD")
	
	xlims!(1,15)
	
	axislegend()
	
	
	fg6
end

# ╔═╡ 3e27b7d8-9dec-4e54-9050-5ca9275d8499
md"""
- The computed periodogram (with this function) is normalized so that the area under the periodogram is equal to the uncentered variance (or average power) of the original signal.

- Since the negative Fourier frequencies (and their associated powers) are discarded, the number of time bins per segment `n` is twice the length of `freq` and `power`.

- The zero frequency is the sum of the signal value, and it not typically used.
"""

# ╔═╡ 817993c0-a1ce-46d2-a213-ed6448d5159f
Markdown.parse("""
Number of data points:   $(latexify(length(noisy)))

Number of data points:   $(latexify(length(psd.freq[2:end])))
""")

# ╔═╡ 7fedf685-66f5-4d46-b52e-99c7305d3f95
md"""
- The power spectrum is a bit noisy. Let's try averaging together power spectra from multiple segments of data using the long time series.

- We want to average periodograms computed each 8 seconds, averaging therefore 1600/8 = 200 periodograms of 256 elements each.
"""

# ╔═╡ 9985012a-ad66-434b-84c2-13eead6a1af7
pf

# ╔═╡ 39538b6b-414d-436a-a1c4-19c6f6754dc7
begin
	apsd = welch_pgram(long_noisy, 256, 0, fs=1/dt)
	
	fg7 = Figure()
	
	ax1fg7 = Axis(fg7[1, 1],
	    yscale = log10,
	    xlabel = "Frequency (Hz)",
	    ylabel = "Power",
	    )
	
	lines!(apsd.freq,apsd.power,color=:blue,label="Averaged PSD")
	
	xlims!(1,15)
	
	axislegend()
	
	fg7
end

# ╔═╡ 549cc9b7-01ff-4639-b8cb-66df220279a9
md"""
- With a clear increase of the S/N.

- Let's try now to compute periodograms using different windows functions.
"""

# ╔═╡ cef2b9a7-701e-4975-8e71-f8a0afff1641
pf

# ╔═╡ b8c5baed-5a58-4380-b941-a7d4035c8d01
begin
	psd_rect = periodogram(noisy; fs=1/dt, window=nothing)
	psd_ham = periodogram(noisy; fs=1/dt, window=hamming)
	psd_tri = periodogram(noisy; fs=1/dt, window=triang)
	psd_cos = periodogram(noisy; fs=1/dt, window=cosine)
	
	
	
	fg8 = Figure()
	
	ax1fg8 = Axis(fg8[1, 1],
	    yscale = log10,
	    xlabel = "Frequency (Hz)",
	    ylabel = "Power",
	    )
	
	lines!(psd_rect.freq,psd_rect.power,label="Rectangular")
	lines!(psd_ham.freq,psd_ham.power,label="Hamming")
	lines!(psd_tri.freq,psd_tri.power,label="Triangular")
	lines!(psd_cos.freq,psd_cos.power,label="Cosine")
	
	
	xlims!(1,5)
	
	axislegend()
	
	fg8
end

# ╔═╡ ee646903-dc54-46d0-b2f5-b30e41a68853
md"""
### Auto and Cross-Correlation
***

- The Cross-correlation of two functions $f(t)$ and $g(t)$ is defined as:

```math
C(\tau)=f\star g=\int\limits_{-\infty}^{\infty}f^*(t)g(t+\tau)dt
```

- The result is a function of the *lag* $τ$ introduced between the two functions, and is often used to estimate the similarity between two different time series, as a function of lag.
- If a common underlying process causes the time variation of intensity at two different electromagnetic bands with differential delays while propagating to the observer, then the cross correlation function of the two time series will exhibit a peak at the corresponding lag, namely the relative delay between the two bands.

- The autocorrelation function is a special case where a function is correlated with itself, which would always show a peak at zero lag.

> Convolution is an operation akin to the Cross-correlation, but the function $g(t)$ in the integrand is inverted to $g(−t)$ before adding the shift.

- A few important properties of cross-correlation include:

```math
[f\star g](\tau) = [g^*\star f^*](-\tau)
```

```math
[f\star g]\star[f\star g] = [f\star f]\star[g\star g]
```

```math
g\star(f\otimes h)  =  [g\star f]\otimes h \\
```

```math
F[f\star g]  =  F(f)\cdot F^*(g)
```

- where $F$ represents the Fourier transform.
"""

# ╔═╡ 454fc15d-ea34-4938-b86e-76f2cb0d5125
md"""
> We now propose a few definitions for completeness, but without extensive discussions.

### Cross-spectra, phase lag spectra, coherence
***

- Given two signals $f(t)$ and $g(t)$ and their respective FTs, $F(ω)$ and $G(ω)$, we define the cross spectrum as:

```math
CS(\omega) = F(\omega)\cdot G^*(\omega)
```

- Analogous to the PSD and the autocorrelation, the cross spectrum between two signals is the Fourier transform of their cross-correlation (and vice-versa).

- In its essence, the cross spectrum of two signals at each frequency is a complex number whose argument represents the phase delay between the signals at that frequency.

- The autocorrelation function may be thought of as the second order correlation:

```math
c_2(\tau)=\langle f(t)f(t+\tau) \rangle
```

- where the angular brackets denote an ensemble average.

- Bispectrum is an extension of the above concept to triple correlations. The third order correlation function:

```math
c_3(\tau_1,\tau_2)=\langle f(t)f(t+\tau_1)f(t+\tau_2)\rangle
```

- that allows one to define the Bispectrum:

```math
B(\omega_1,\omega_2)=\int\limits_{-\infty}^{\infty}\int\limits_{-\infty}^{\infty} c_3(\tau_1,\tau_2) e^{-i(\omega_1\tau_1+\omega_2\tau_2)}\,d\tau_1\,d\tau_2
```
"""

# ╔═╡ 6e178c7b-1846-4390-b842-b4d303688bd8
md"""
## Power Spectrum statistics
***

- The first thing we need to know is the probability distribution of the noise power.

- We now assume that it is additive and independent of the frequency (i.e. noise is “white”):

```math
P_j = P_{j,noise} + P_{j,signal}
```

- The “null hypothesis” is that the periodogram is consistent with pure noise.

- Let’s remind us that If we have $x_k ≡ y_k + z_k$ and $b_j$ and $c_j$ are the FT of $y_k$ and $z_k$, respectively, we have $a_j = b_j + c_j$.

- This does not hold for power spectra unless the signal are uncorrelated random noise:

```math
|a_j|^2 = |b_j + c_j|^2 = |b_j|^2 + |c_j|^2 + {\rm cross\ terms}
```

- For a wide range of types of noise, $P_{j,noise}$ follows a $\chi^2$ distribution with 2 degrees of freedom (but at the Nyquist frequency, where dof is 1).

- For other noise distributions (Poisson, etc.), for large N, due to the central limit theorem the Fourier coefficients $A_j$ and $B_j$ become still normal.

- This suggests a simple consistency test: compute the standard deviation in each frequency bin and divide by the mean power. Results should be (if the hypotheses hold) close to 1.

- In practice one finds that noise powers are nearly always $\chi^2$ distributed, not only for Poisson noise, but also for many other types of noise.

- With the Leahy normalization, the probability for $P_{j,noise}$ to exceed a given threshold is given by:

```math
Prob(P_{j,noise} > P_{threshold}) = Q(P_{threshold} | 2) \quad (j = 1, N/2-1)
```

```math
Q(\chi^2|\nu) \equiv \left[2^{\nu/2} \Gamma(\frac{\nu}{2})\right]^{-1} \int_{\chi^2}^\infty t^{\frac{\nu}{2}-1} e^{-\frac{t}{2}} dt
```

- where $\nu$ is the number of dof and $\Gamma$ is the gamma function, the generalization of factorial to real and complex numbers [$\Gamma(n) = (n-1)!$]

- For $\nu = 2$:

```math
Q(\chi^2|2) = \frac{1}{2} \int_{\chi^2}^\infty e^{-\frac{t}{2}} dt = e^{-\frac{\chi^2}{2}}
```

"""

# ╔═╡ 83aece6e-cc50-4ce4-b9b5-f518342620e5
md"""
- Power spectra are unavoidably very noisy.

    - Standard deviation of noise power is equal to the mean value ($σ_{P_j} = < P_j > = 2$).

- More interesting, this cannot be improved increasing the number of data points (i.e. the length of the time series). This merely increases the number of powers.

- One can decrease the large variance rebinning the power spectrum and/or dividing the data in multiple segments of equal length. This of course degrade the frequency resolution.  

    - If the number of segments is large the power statistics tends to become Normal.

- Let’s define  confidence detection level as the power with only $ε$ probability to be exceeded by noise.

- This holds for a single frequency. If your spectrum consists of $N_{\rm trial}$ (independent) frequencies, the confidence detection level decreases to take into account the multiple trials:

$$(1 - \epsilon)^{N_{\rm trial}} \sim 1 - \epsilon N_{\rm trial} \quad {\rm for\ } \epsilon << 1$$

- In general if noise is not Poissonian or Gaussian noise power spectrum will not be flat anymore.

    - However, often noise powers still follow a $\chi^2$ distribution with 2 dof, but with a different normalisation (in general depending on $j$).
"""

# ╔═╡ 06df9913-445f-4170-b110-13ec4208d940
cm"""
### The Likelihood for Periodograms
***

- The ``\chi^2`` distribution defines a sampling distribution or likelihood, i.e. the probability distribution of observing a given data set given some underlying (true, unknown) power spectrum.

- If we define a model power ``S_j(θ)`` at frequency ``ν_j``, specified by a set of parameters ``θ``, we can then compute the probability of having observed periodogram power ``P_j`` at that same frequency:

```math
p(P_j|S_j(θ)) = \frac{1}{S_j(θ)} e^{-\frac{P_j}{S_j(θ)}}
```

- The likelihood (also known as *Whittle likelihood*) for a periodogram over ``N/2`` observed powers ``P_j`` is then defined as the product of individual probabilities for each frequency ``ν_j``.

- One generally (and equivalently), defines the logarithm of the likelihood as the sum of logarithm of all probabilities, such that:

```math
\log(\mathcal{L}(\theta)) = \sum_{j=1}^{N/2} \log(p(P_j|S_j(θ))) =  -\sum_{j=1}^{N/2} ( \log(S_j(θ)) + \frac{P_j}{S_j(θ)})
```

- A likelihood for averaged periodograms can be derived from the ``\chi^2_{2LM}/2ML`` sampling distribution for periodograms averaged over ``L`` independent segments and ``M`` independent neighbouring frequencies by:

```math
\log(\mathcal{L}_{avg}(\theta)) = -2ML\sum_{j=1}^{N/2} \left[ \frac{P_j}{S_j(θ)}) + \log(S_j(θ)) + (\frac{1}{ML} - 1) \log(P_j) + c(2ML) \right]
```

- where ``c(2ML)`` is a factor independent of ``P_j`` or ``S_j``, and thus unimportant to the parameter estimation problem considered here.

"""

# ╔═╡ 9a17f40a-2aad-4b52-96a4-7c275fda9cee
# ╠═╡ show_logs = false
md"""
### Noise color
***

- Fourier analysis (and related techniques) has proven to be very effective in identifying periodic behaviours.

- In astrophysics, however, we often have to deal with phenomena too long for having a reliable coverage (decades or more) and with only approximately cyclical behaviours (quasi-periodicities).

- More important, in order to compute the statistical significance of any possible periodicity, one needs to properly model the noise affecting a time series.

$(LocalResource("Pics/colnoise.jpg"))

- "Coloured" noise directly affect a time-series shape:

$(LocalResource("Pics/redblunoise.png"))
"""

# ╔═╡ a76024f5-7355-4d57-9167-ed522a76ff51
md"""
## Final considerations
***

- Fourier analysis reveals nothing about the evolution in time, but rather reveals the variance of the signal at different frequencies.

- The classical periodogram is an estimator of the spectral density, i.e. the Fourier transform of the autocovariance function.

- Fourier analysis has restrictive assumptions: an infinitely long data of equally-spaced observations; homoscedastic Gaussian noise with purely periodic signal of sinusoidal shape.

- The classical periodogram is not a good estimator, it is “inconsistent” because the number of parameters grows with the number of datapoints.

- The DFT and its probabilities depends on several strong assumptions that are rarely achieved in real astronomical data: evenly spaced data of infinite duration with a high sampling rate (Nyquist frequency), Gaussian noise, single frequency periodicity with sinusoidal shape and stationary behavior.

- Each of these constraints is often violated in various astronomical problems. Data spacing may be affected by daily/monthly/orbital cycles. Periods may be comparable to the sampling time. Several periods may be present (e.g. helioseismology). Shape may be non-sinusoidal (e.g. elliptical orbits, eclipses, recurrent flares). Periods may not be constant (e.d. QPOs in accretion disks).
"""

# ╔═╡ ddfd682b-56a7-49c9-883a-0d348c9cb8c8
md"""
## Reference & Material

Material and papers related to the topics discussed in this lecture.

- [Belloni & Bhattacharya (2022) - "Basics of Fourier Analysis for High-Energy Astronomy”](https://ui.adsabs.harvard.edu/abs/2022hxga.book....7B/abstract)
- [van der Klis (1988) - "Fourier techniques in X-ray timing"](https://ui.adsabs.harvard.edu/abs/1989ASIC..262...27V/abstract)
"""

# ╔═╡ 95ec0443-95e3-4986-b170-7589328246b2
md"""
## Further Material

Papers for examining more closely some of the discussed topics.

- [Vaughan (2010) - "A Bayesian test for periodic signals in red noise"](https://ui.adsabs.harvard.edu/abs/2010MNRAS.402..307V/abstract)
- [Barret & Vaughan (2012) - "Maximum Likelihood Fitting of X-Ray Power Density Spectra: Application to High-frequency Quasi-periodic Oscillations from the Neutron Star X-Ray Binary 4U1608-522](https://ui.adsabs.harvard.edu/abs/2012ApJ...746..131B/abstract)
- [Covino eta al. (2019) - "Gamma-ray quasi-periodicities of blazars. A cautious approach"](https://ui.adsabs.harvard.edu/abs/2019MNRAS.482.1270C/abstract)
"""

# ╔═╡ 22c6e22a-4b5d-47cd-b4b1-97543a7c9d74
md"""
### Credits
***

This notebook contains material obtained from [https://www.tutorialspoint.com/power-spectral-density-psd-and-autocorrelation-function#](https://www.tutorialspoint.com/power-spectral-density-psd-and-autocorrelation-function#) and from [https://ipython-books.github.io/101-analyzing-the-frequency-components-of-a-signal-with-a-fast-fourier-transform/](https://ipython-books.github.io/101-analyzing-the-frequency-components-of-a-signal-with-a-fast-fourier-transform/).
"""

# ╔═╡ 27d77a9f-afa4-4b79-87f8-f8a92b87381e
cm"""
## Course Flow

<table>
  <tr>
    <td>Previous lecture</td>
    <td>Next lecture</td>
  </tr>
  <tr>
    <td><a href="./open?path=Lectures/Lecture - Statistics Reminder/Lecture-BayesianReminder.jl">Lecture about Bayesian statistics</a></td>
    <td><a href="./open?path=Lectures/Science Case - Sunspot Number/Lecture-SunspotNumber.jl">Science case about Sunspot number</a></td>
  </tr>
 </table>


"""

# ╔═╡ 321154fd-5092-4d93-ba82-1bcde10efcb5
md"""
**Copyright**

This notebook is provided as [Open Educational Resource](https://en.wikipedia.org/wiki/Open_educational_resources). Feel free to use the notebook for your own purposes. The text is licensed under [Creative Commons Attribution 4.0](https://creativecommons.org/licenses/by/4.0/), the code of the examples, unless obtained from other properly quoted sources, under the [MIT license](https://opensource.org/licenses/MIT). Please attribute the work as follows: *Stefano Covino, Time Domain Astrophysics - Lecture notes featuring computational examples, 2025*.
"""

# ╔═╡ Cell order:
# ╟─d9329a8d-b07c-4207-93a6-1668da23e296
# ╟─3307644d-e875-4f15-a140-a41f7ca82a8f
# ╟─419e9d2c-f5df-49a3-b0ce-9804aacda4b7
# ╟─7209ecae-fba8-44d7-940e-1ecc12e6700a
# ╟─6a59a8e6-39fc-44b3-9921-8976c677f4b1
# ╟─95ee75d5-3112-42b6-83aa-29e639ac6eb0
# ╟─b175b388-911f-4862-afdb-7449fec2bf9e
# ╟─882fc480-b132-45a8-906a-db157059b92c
# ╟─231efd02-fbde-4c81-9577-7fea1ac07e88
# ╟─bc82a078-fea7-4a00-b38c-a849fb760594
# ╟─3bdf3bea-7e3c-4f09-98f7-bdbc16ea2880
# ╟─0dbb5f0e-292d-4d8a-bb83-7954e4a46f29
# ╟─875732fc-5644-4c06-8f8c-39dc30f1d3b4
# ╟─56081488-4ade-436a-991e-d7138368edf1
# ╟─6d88e888-8f08-47ab-8906-c7b4b86b2588
# ╟─99f8ea18-5807-482e-8516-e2bdaf1546e6
# ╟─e087bf1e-dd1a-41f3-8154-bfb39325d905
# ╟─de228ddb-cacc-456f-b6c7-464e81d1e1e8
# ╟─e82d615c-5201-4efb-8b77-37a5aa636f39
# ╟─58d43f9c-e7e9-4a7d-8471-1beca3568a1e
# ╟─3d75fecb-157b-408a-aa32-2d2096a766ba
# ╟─2e2e09aa-6023-46a9-b816-bb9de2b321c1
# ╟─0148c42e-24fa-486e-8c61-eae25e772bd8
# ╟─7709a017-0c80-4399-9751-bf07b80f0b5e
# ╟─b49f1cda-582a-4d43-9e0a-a0e6adf85e78
# ╟─c601e37c-c08d-4ce4-8843-6fd52bd1e812
# ╟─aff12d14-4803-480e-aab0-6d33917304b3
# ╟─6f29671a-048a-49e1-b7f9-e5d9cd733a0c
# ╟─b8a2074a-490e-4e92-ba59-3735184d7c91
# ╟─40fe7f7d-8d30-40f1-b6fa-932374c93380
# ╟─1eb98eee-8373-4fce-b40f-3118d096f082
# ╟─0f5adef3-5de0-4fa3-b4ca-8ee59968047e
# ╟─1b13be55-5153-4d53-ba09-92832f67a448
# ╟─bf2f9df2-7df7-43d2-926d-7a561be79c0e
# ╟─bea9ec97-57a7-45a4-9a29-ccca274fd5dc
# ╟─c1363e2a-d8b4-4be9-8394-435231b62177
# ╟─f5fea622-f431-4da6-af3f-548fd10d906d
# ╟─43caa1a2-3083-414d-9681-4153ade8a92b
# ╟─dd78a7e2-f63e-4e84-af31-286d979074f5
# ╟─95095834-abd5-43e6-aad0-cdfa402b0366
# ╟─1cfc40d3-24a3-4a5c-8538-61c13542b403
# ╟─e8786a24-4391-40ad-a9ba-a0cb3f7bb2b0
# ╟─39e1ee25-a2f4-4e16-9fec-4a4e9c2c653e
# ╟─136b29c5-ac92-4e32-878e-3ad99216f78d
# ╟─d490adbd-376a-403b-adc3-3ab4d4e65bc5
# ╟─848e6f85-e3b9-4928-aed0-b316f97d18db
# ╟─8d5c7719-647a-4e7e-905f-43051bb26716
# ╟─a9964f9d-1f3a-493b-9ab8-a8933ff51c72
# ╟─a0392d53-9e0d-4d70-a9a5-b99de7884f9d
# ╟─eae97444-2431-448c-abe6-c14aa64fa409
# ╟─8d3eb863-5cb0-4221-8fc8-c48361615e95
# ╟─0b0b8578-4c67-463a-bd53-b3e6bd1f9712
# ╟─8e458457-f940-460a-8638-debe1f94defc
# ╟─b2e2bace-7e50-41f0-87fb-780737ff6616
# ╟─1112b278-802d-4611-8954-2795db178e88
# ╟─19a6c251-adc1-422d-a702-20ea9b971449
# ╟─a8b99222-18a7-4b47-8f98-8961a708ec57
# ╟─77b0596d-8c4a-4d3d-9bd8-54d3fde618c8
# ╟─d542c639-1c93-4edd-9132-facd18fc89fb
# ╟─b96525bf-90e5-47b3-b305-1db0ad675c2c
# ╟─ae8a4ebf-da8d-4778-ad9a-87e3aab72399
# ╟─b7213b38-21ac-4fb5-ab53-694d3ac2fda9
# ╟─78dc57ce-1cbd-4f9c-9741-54bf430b9186
# ╟─25fcf199-7da4-4632-a2fa-d0131f58d533
# ╟─3e27b7d8-9dec-4e54-9050-5ca9275d8499
# ╟─817993c0-a1ce-46d2-a213-ed6448d5159f
# ╟─7fedf685-66f5-4d46-b52e-99c7305d3f95
# ╟─9985012a-ad66-434b-84c2-13eead6a1af7
# ╟─39538b6b-414d-436a-a1c4-19c6f6754dc7
# ╟─549cc9b7-01ff-4639-b8cb-66df220279a9
# ╟─cef2b9a7-701e-4975-8e71-f8a0afff1641
# ╟─b8c5baed-5a58-4380-b941-a7d4035c8d01
# ╟─ee646903-dc54-46d0-b2f5-b30e41a68853
# ╟─454fc15d-ea34-4938-b86e-76f2cb0d5125
# ╟─6e178c7b-1846-4390-b842-b4d303688bd8
# ╟─83aece6e-cc50-4ce4-b9b5-f518342620e5
# ╟─06df9913-445f-4170-b110-13ec4208d940
# ╟─9a17f40a-2aad-4b52-96a4-7c275fda9cee
# ╟─a76024f5-7355-4d57-9167-ed522a76ff51
# ╟─ddfd682b-56a7-49c9-883a-0d348c9cb8c8
# ╟─95ec0443-95e3-4986-b170-7589328246b2
# ╟─22c6e22a-4b5d-47cd-b4b1-97543a7c9d74
# ╟─27d77a9f-afa4-4b79-87f8-f8a92b87381e
# ╟─321154fd-5092-4d93-ba82-1bcde10efcb5
